/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_MATCHING_INTERFACE_H_
#define _SCE_MATCHING_INTERFACE_H_
#include "matching_defines.h"
#include "message_service_interface.h"

using namespace sce::Toolkit::NP::Utilities;
using namespace sce::Toolkit::NP;

namespace sce {
	namespace Toolkit{
		namespace NP{

	/// @brief
	/// The namespace containing matching functionality.
	///
	/// The namespace containing matching functionality.
			namespace Matching {

	/// @brief
	/// The matching interface allows users to find other players and game sessions for online play.
	///
	/// The matching interface allows users to find other players and game sessions for online play.
	class Interface
	{
	public:

		/// @brief
		/// Registers attributes used by the session. 
		///
		/// Registers the session attributes which are going to be used by the application during session 
		/// creation or searching. 
		/// For correct working of session attributes and filtering, the application needs to register the session attributes when it initializes. 
		/// The application should not re-register attributes once specified. Once registered, the attributes are maintained throughout the
		/// life cycle of the application. 
		///
		/// @param sessionAttributes       A pointer to the memory address which contains information about the attributes to register.
		/// @param numParameters		   The number of attributes to be registered.
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS									The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_MATCHING_INVALID_PARAMETERS				The operation failed because information about the attributes was missing. 
		/// @retval SCE_TOOLKIT_NP_MATCHING_FAIL_TO_REGISTER_PARAMETERS    The operation failed because the application tried to register more attributes than it can. For more information on this please refer to the <em>%NP %Toolkit Library Overview</em>.
		///
		/// @note
		/// If a session attribute which the application failed to register is referenced during creation, searching or modifying a session, then SessionInformation
		/// will not contain any information on the attribute. 
		static int registerSessionAttributes(const RegisterSessionAttribute *sessionAttributes, int numParameters);
		
		/// @brief
		/// Creates a session room.
		/// 
		/// Creates a session room on the %NP matching server. The session creation process is kicked off on a different thread. 
		///
		/// Non-Blocking (<c><i>async</i></c> = true)
		/// If the process was not able to kick off, the function returns an appropriate error code. 
		/// On completion of the process the application is notified using a <c>matchingSessionCreated</c> Event. The application can then verify whether there was an error during 
		/// the process or whether the process was successful by using the <c>hasError()</c> or <c>hasResult()</c> method of the <c>Future</c> object.
		/// If an error has occurred, the application can get the error code using the <c>getError()</c> method.
		///
		/// Blocking (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application can retrieve session information using the <c>get()</c> method of the <c>Future</c> object.
		///
		///	@param	sessionRequest			A structure that describes the session details. This will be assigned to a room on successful creation.
		///	@param	sessionInformation		Output. Receives the session information upon successful creation of the session. 
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS								The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_NOT_INITIALISED						The operation failed because the %NP %Toolkit library is not initialized.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY				The operation failed because the matching service is busy processing a previous request.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_ALREADY_ACTIVE		The operation failed because the user is already in a session. They must leave a session in order to join or create a new session.
		/// @retval other											An %NP Library Error Code. 
		///
		/// @note
		/// If this function is called from the main thread, it should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid 
		/// until the callback of the event is processed. 
		///
		/// @see
		/// searchSessions,
		/// joinSession
		static int createSession(const CreateSessionRequest *sessionRequest, Future<SessionInformation> *sessionInformation, bool async = true);
		
		/// @brief
		/// Searches for a session. 
		/// 
		/// Searches for a session on the %NP matching server.  On calling, this function 
		/// kicks off a process that searches for a session on a %NP %Toolkit library thread.
		///
		/// Non-Blocking Process (<c><i>async</i></c> = true)		
		/// If the process was not able to kick off, the function returns an appropriate error code. 
		/// On completion of the process, the application is notified by a <c>matchingSessionSearchCompleted</c>
		/// Event. The application can then verify whether there was an error during 
		/// the process or whether the process was successful by using the <c>hasError()</c> or <c>hasResult()</c> method of the <c>Future</c> object.
		/// If an error has occurred, the application can get the error code using the <c>getError()</c> method.
		///
		/// Blocking Process (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application can retrieve the result using the 
		/// <c>get()</c> method of the <c>Future</c> object. 
		///
		///	@param	searchRequest			Specifies the type of a session to look for.
		///	@param	sessionList				Output. Receives the results of the search. 
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS					The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_NOT_INITIALISED			The %NP %Toolkit library is not initialized.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY	The matching service is busy processing a previous request.
		/// @retval other								An %NP Library Error Code. 
		///
		/// @note
		/// If calling from main thread, the function should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid 
		/// until the callback of the event is processed.
		///
		/// @see
		/// createSession(), 
		/// joinSession()
		static int searchSessions(const SearchSessionsRequest *searchRequest,Future<SessionList> *sessionList, bool async = true);
 		
		/// @brief
 		/// Joins a specific session.
		///
		/// Joins a session on the %NP matching server. 
		/// This function kicks off a process that requests to join a session on a different thread.
		///
		/// Non-Blocking Process (<c><i>async</i></c> = true)	
		/// If the process was not able to kick off, then the function returns an appropriate 
		/// error code.
		/// On completion of the process the application is notified by a <c>matchingSessionJoined</c> Event. 
		/// The application can then verify whether there was an error during the process 
		/// or whether the process was successful by using the <c>hasError()</c> or <c>hasResult()</c> method
		/// of the <c>Future</c> object. If an error has occurred, the application can get the  
		/// error code using the <c>getError()</c> method.
		///
		/// Blocking Process (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application can retrieve the result using the 
		/// <c>get()</c> method of the <c>Future</c> object. 
		///
		///	@param	sessionJoinRequest		A structure which contains information about session to be joined.
		///	@param	sessionInformation		Output. Receives the session information when a session is successfully joined. 
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS								The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_NOT_INITIALISED						The operation failed because the %NP %Toolkit library is not initialized.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY				The operation failed because the matching service is busy processing a previous request.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_ALREADY_ACTIVE		The operation failed because the user is already in a session. They must leave a session in order to join or create a new session.
		/// @retval other											An %NP Library Error Code. 
		///
		/// @note
		/// If calling from main thread, the function should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid
		/// until the callback of the event is processed.
		///
		/// @see
		/// searchSessions, 
		/// createSession				
		static int joinSession(const JoinSessionRequest *sessionJoinRequest,Future<SessionInformation> *sessionInformation, bool async = true);

		/// @brief
		/// Modifies a specific session.
		///
		/// Modifies session(room) information on the %NP matching server. 
		/// This function kicks off a process that requests to modify a session on a different thread.
		///
		/// Non-Blocking Process (<c><i>async</i></c> = true)	
		/// If the process was not able to kick off, then the function returns an appropriate 
		/// error code.
		/// On completion of the process the application is notified by a <c>matchingSessionModified</c> Event. 
		/// The application can then verify whether there was an error during the process 
		/// or whether the process was successful by using the <c>hasError()</c> or <c>hasResult()</c> method
		/// of the <c>Future</c> object. If an error has occurred, the application can get the  
		/// error code using the <c>getError()</c> method.
		///
		/// Blocking Process (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application should update the session information by calling updateSession() and specifying <c>matchingSessionModified</c> for the
		/// <c><i>userEvent</i></c> parameter the function.
		///
		/// @param 	sessionDesc				The session attributes to modify. 
		/// @param  processResult			Output. Receives the result of this process when the session has been successfully modified.
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful.							 
		/// 
		/// @note
		/// Only the session owner can modify the session information with the exception of #SCE_TOOLKIT_NP_SESSION_MEMBER_ATTRIBUTE. If any other session member tries to modify an attribute of a session other than #SCE_TOOLKIT_NP_SESSION_MEMBER_ATTRIBUTE, 
		/// the function will immediately return an error code.
		/// If calling from main thread, the function should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid
		/// until the callback of the event is processed.
		///
		/// @see
		/// searchSessions, 
		/// createSession
		static int modifySession(const ModifySessionRequest *sessionDesc, Future<int>*processResult, bool async = true );

		/// @brief
		/// Searches for a session and join the first one available. 
		/// 
		/// This function searches for a session and joins the first one available on the %NP matching
		/// server. This function kicks off a process that searches for a session on a 
		/// %NP %Toolkit library thread. 
		///
		/// Non-Blocking Process (<c><i>async</i></c> = true)	
		/// If the process was not able to kick off, then the function 
		/// returns an appropriate error code.	
		/// On successful completion of the process the application is notified 
		/// by a <c>matchingSessionJoined</c> event. The application can then verify whether 
		/// there was an error during the process or whether the process was successful by 
		/// using the <c>hasError()</c> or <c>hasResult()</c> method of the <c>Future</c> object.
		/// If an error has occurred, the application can get the error code using the <c>getError()</c>
		/// method.
		///
		/// Blocking Process (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application can retrieve the result 
		/// using the <c>get()</c> method of the <c>Future</c> object.
		///
		///
		///	@param	searchCriteria			Specifies the type of a session to look for. Set to <c>NULL</c> if not looking for a specific session.  
		///	@param	sessionInformation		Output. Receives the session information when a session is successfully joined. 
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS								The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_NOT_INITIALISED						The operation failed because the %NP %Toolkit library is not initialized.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY				The operation failed because the matching service is busy processing a previous request.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_ALREADY_ACTIVE		The operation failed because the user is already in a session. They must leave a session in order to join or create a new session.
		/// @retval other											An %NP Library Error Code. 
		///
		/// @note
		/// If calling from main thread, the function should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid 
		/// until the callback of the event is processed.
		///
		/// @see
		/// searchSessions,
		/// createSession, 
		/// joinSession
		static int quickSession(const SearchSessionsRequest *searchCriteria,Future<SessionInformation> *sessionInformation, bool async = true);
		
		/// @brief
		/// Leaves a currently joined/created session.
		/// 
		/// Leaves a currently joined/created session. This request is processed on a different
		/// thread. 
		///
		/// Non-Blocking Process (<c><i>async</i></c> = true)	
		/// If the process was not able to kick off, then the function returns an 
		/// appropriate error code.
		///	On successful completion of the process the application is notified by a 
		/// <c>matchingSessionLeft</c> Event. The application can then verify whether 
		/// there was an error during the process or whether the process was successful by 
		/// using the <c>hasError()</c> or <c>hasResult()</c> method of the <c>Future</c> object.
		/// If an error has occurred, the application can get the error code using the <c>getError()</c> 
		/// method.
		/// 
		/// Blocking Process (<c><i>async</i></c> = false)
		/// The function blocks until a result is returned. If the function is successful, it returns #SCE_TOOLKIT_NP_SUCCESS;
		/// otherwise an appropriate error code is returned.
		///
		/// On successful completion of the operation, the application can retrieve the result 
		/// using the <c>get()</c> method of the <c>Future</c> object.
		/// 
		///	@param	leavingSession			A pointer to the session the user wants to leave. Set to <c>NULL</c> if the session is not known. 
		///	@param	processResult			Output. Receives the result of this process when the session has been successfully left.
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS							The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_NOT_INITIALISED					The operation failed because the %NP %Toolkit library is not initialized.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY			The operation failed because the matching service is busy processing a previous request.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_DOES_NOT_EXIST	The operation failed because the session which the user is trying to leave does not exist.
		/// @retval other										An %NP Library Error Code. 
		///
		/// @note
		/// If calling from main thread, the function should always be non-blocking.
		/// If calling asynchronously, the <c>Future</c> object should be valid 
		/// until the callback of the event is processed.
		/// Even when an application receives a <c>matchingSessionError</c> Event, the session will still be destroyed
		/// and therefore no longer be valid.
		///
		/// @see
		/// searchSessions,
		/// createSession,
		/// joinSession
		static int leaveSession(const SessionInformation *leavingSession, Future<int> *processResult, bool async = true);
		
		/// @brief
		/// Updates the current session information.
		/// 
		/// Updates the current session information. This function should be called when an application receives a <c>matchingSessionUpdate</c> or <c>matchingSessionModified</c> Event.
		/// 
		///	@param	currentSession									Input/Output. The current session information structure to be updated.			
		/// @param  event											Output. Receives the %NP Matching2 event that caused the session to be updated.
		/// @param  userEvent										Input. The event for which the session update is called. This could be either a <c>matchingSessionUpdate</c> or <c>matchingSessionModified</c> Event.
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS								The operation was successful.							 
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_KICKEDOUT			The operation failed because the user has been kicked out of the matching session. 
		/// @retval SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY				The operation failed because the matching service is busy processing a previous request.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_DOES_NOT_EXIST		The operation failed because the session the user was trying to update does not exist.
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_ROOM_DESTROYED		The operation failed because the session the user was in has been destroyed.
		///
		///
		/// @note
		/// On receiving an error, the application should clear the current session.
		///
		/// @note
		/// This operation should be called from the same thread on which the callback was received (the %NP %Toolkit thread).
		///
		/// @see
		/// searchSessions,
		/// createSession,
		/// joinSession
		static int updateSession(SessionInformation *currentSession,SceNpMatching2Event * event, Event::UserEvent userEvent);

		/// @brief
		/// Updates the session attributes of the current session.
		/// 
		/// Updates the session attributes of the current session.
		/// 
		///	@param	attributeRequest		Input. The session attribute to be updated.			
		/// @param  processResult			Output. Receives the result of this process when the session has been successfully updated.
		/// @param  async					A flag that indicates whether the function is non-blocking or blocking. 
		///									Defaults to true so the function is non-blocking by default.
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful.							 
		///
		/// @see
		/// searchSessions,
		/// createSession,
		/// joinSession
		static int updateSessionAttribute(sce::Toolkit::NP::UpdateAttributeRequest* attributeRequest, sce::Toolkit::NP::Utilities::Future<int>*processResult, bool async = true );
		/// @brief
		/// Sends a session invite to a friend of the user.
		///
		/// Sends a session invite to a friend of the user.
		///
		///	@param	currentSession			A pointer to the session the user currently is in. 
		/// @param	msg						The user specified message.
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS								The operation was kicked off successfully.	
		/// @retval SCE_TOOLKIT_NP_MATCHING_SESSION_DOES_NOT_EXIST		The operation failed because the session that an invite was sent for is invalid.
	
		static int inviteToSession(const SessionInformation *currentSession, const InviteMessage* msgDetail);


		/// @brief
		/// Joins the user to a session that they were invited to.
		///  
		/// Joins the user to a session that they were invited to.
		/// When the operation has completed an event callback will be generated, and the application can retrieve the result using the 
		/// <c>get()</c> method of the <c>Future</c> object. 
		///
		/// @param attachment					A pointer to the message attachment which was retrieved using MessagingServiceInterface::retrieveMessageAttachment(). 		
		/// @param sessionInformation			Output. Receives the session information when a session is successfully joined. 
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS						The operation was kicked off successfully.							 
		/// @retval SCE_TOOLKIT_NP_MESSAGE_ATTACHMENT_INVALID	The operation failed because the message attachment type is not supported by the %NP %Toolkit library.
		///
		/// @note 
		/// This function is asynchronous.
		///
		/// @note 
		/// The <c>Future</c> object should be valid until the callback of the event is processed.
		///
		///
		/// @see
		/// inviteToSession	
		static int joinInvitedSession(const MessageAttachment *attachment,Future<SessionInformation> *sessionInformation );
	
		/// @brief
		/// Registers a callback which will be called for room messages. 
		///
		/// Registers a callback which will be called for room messages. 
		///
		/// @param cbfunc						The room message callback function to register. 
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS			The operation was successful.							 
		/// 
		/// @see
		/// inviteToSession
		int registerRoomMessageCallback(SessionMessageCallback cbfunc);

		/// @brief
		/// Sends room/chat messages to the room members.
		///  
		/// Sends room/chat messages to the room members. When the operation has been completed, the callback registered with registerRoomMessageCallback()
		/// is called with the appropriate %NP matching2 event and room message event (corresponding to #SCE_TOOLKIT_NP_MATCHING_INVALID_ROOM_MESSAGE).
		///
		/// @param msgFlag						The room message type.
		/// @param msg							The message data.
		/// @param member						The list of members to send the message to. To broadcast to all members, specify an empty list.
		/// @param numMembers					The number of members in the list.	
		///
		/// @retval SCE_TOOLKIT_NP_SUCCESS		The operation was successful.							 
		///
		/// @see
		/// inviteToSession
		static int sendRoomMessage(SceToolkitNpRoomMessageFlag msgFlag,std::string msg, SceNpMatching2RoomMemberId* member, uint32_t numMembers);

	};
			}//end matching
		}//end NP
	}//end Toolkit
}//end sce
#endif
